import math

# Define the HDGL Machine State
class HDGLMachine:
    def __init__(self):
        # Upper Field: Expansion Primitives
        self.upper_field = {
            "prism_state": 105.0,
            "recursion_mode": 99.9999999999,
            "positive_infinite": 9.9999999999,
            "P3": 4.2360679775,
            "pi": 3.1415926535,
            "phi_power_phi": 2.6180339887,
            "phi": 1.6180339887
        }

        # Analog Dimensionality Primitives (Active State)
        self.analog_dims = {
            "D8": 8.0,
            "D7": 7.0,
            "D6": 6.0,
            "D5": 5.0,
            "D4": 4.0,
            "D3": 3.0,
            "D2": 2.0,
            "D1": 1.0,
            "dim_switch": 1.0,
            "P4": 6.8541019662,
            "P5": 11.0901699437,
            "P6": 17.94427191,
            "P7": 29.0344465435
        }

        # The Void
        self.void = 0.0

        # Lower Field: Contraction Primitives
        self.lower_field = {
            "negative_infinite": 0.0000000001,
            "inv_P7": 0.0344465435,
            "inv_P6": 0.05572809,
            "inv_P5": 0.0901699437,
            "inv_P4": 0.1458980338,
            "inv_P3": 0.2360679775,
            "inv_phi_power_phi": 0.3819660113,
            "inv_phi": 0.6180339887
        }

        # Current state
        self.current_state = self.void
        self.dimension = self.analog_dims["dim_switch"]
        self.recursion_active = False

    def toggle_recursion(self):
        """Toggle recursion mode based on recursion_mode value."""
        self.recursion_active = not self.recursion_active
        print(f"Recursion mode: {'ON' if self.recursion_active else 'OFF'}")

    def compute_harmonic_state(self, t):
        """Compute the harmonic state as a superposition of primitives at time t."""
        # Combine upper and lower fields with a harmonic oscillation
        state = self.void
        for key, value in self.upper_field.items():
            state += value * math.sin(t * self.upper_field["phi"])
        for key, value in self.lower_field.items():
            state += value * math.cos(t * self.lower_field["inv_phi"])
        for key, value in self.analog_dims.items():
            state += value * math.sin(t * self.upper_field["phi_power_phi"])

        # Apply dimensionality switch
        if self.dimension == 1.0:
            state *= math.sin(t * self.upper_field["pi"])
        else:
            state *= math.cos(t * self.upper_field["pi"])

        return state

    def step(self, t):
        """Advance the machine state for time step t."""
        if self.recursion_active:
            # Recursive state update using Fibonacci braids
            self.current_state = self.compute_harmonic_state(t)
            self.current_state *= self.analog_dims["P7"] / self.lower_field["inv_P7"]
        else:
            # Non-recursive update
            self.current_state = self.compute_harmonic_state(t)
        return self.current_state

    def run_simulation(self, steps=10, dt=0.1):
        """Run a simulation and return the list of states."""
        states = []
        print("Starting HDGL Machine Simulation")
        for i in range(steps):
            t = i * dt
            state = self.step(t)
            states.append(state)
            print(f"Time {t:.2f}: State = {state:.6f}")
        print("Simulation complete")
        return states

# Bootstrap and run the machine
if __name__ == "__main__":
    machine = HDGLMachine()
    machine.toggle_recursion()
    simulation_states = machine.run_simulation(steps=10, dt=0.1)

    if simulation_states:
        mean_state = sum(simulation_states) / len(simulation_states)
        print(f"\nMean State Over Time: {mean_state:.6f}")